///////////////////////////////////////////////////////////////////////////////////////
//
// MonoComm
//
// 09/05/07		Hounddog	Connection statistics
// 08/05/07		Hounddog	Disconnect recovering
// 03/05/07		Hounddog	Sporadic fault attribute
// 21/04/07		Hounddog	Messages suspension
// 25/03/07		Hounddog	MC_EMULATE_COMM
// 25/02/07		Hounddog	Errors recovering
// 09/02/07		Hounddog	Initial implementation
//

#include "Precompiled.h"

#include "Public\MonoComm.h"
#include "VagComm.h"

///////////////////////////////////////////////////////////////////////////////////////

#undef MC_EMULATE_COMM				// define to emulate ECU commmunication and test GUI
#undef MC_EMULATE_COMM_ERRORS		// define to emulate communication erros and test recovering
#undef MC_SWAP_RTS_DTR

///////////////////////////////////////////////////////////////////////////////////////

// Port opening / closing

MC_STATUS McOpenPort(LPCSTR pszPortName, DWORD dwBaudRate);
void McClosePort();

// High-level comm

MC_STATUS McSetupConnection(BYTE bAddress);
MC_STATUS McReceiveBlockInfoRes(LPSTR** pppszDescription);

MC_STATUS McSendBlock(const VAG_BLOCK* pvb);
MC_STATUS McReceiveBlock(VAG_BLOCK** ppvb);

MC_STATUS McReconnect();

VAG_BLOCK* McAllocBlock(BYTE bSize);
void McFreeBlock(VAG_BLOCK* pvb);

// Low-level comm

MC_STATUS McSetPortTimeout(DWORD dwReadAddTimeout, DWORD dwReadMultTimeout);

MC_STATUS McWritePort0(DWORD dwCountToWrite, const VOID* pvBuffer);
MC_STATUS McWritePort1(DWORD dwCountToWrite, const VOID* pvBuffer);
MC_STATUS McWritePort2(DWORD dwCountToWrite, const VOID* pvBuffer);

MC_STATUS McReadPort0(DWORD dwCountToRead, VOID* pvBuffer, DWORD& rdwCountRead);
MC_STATUS McReadPort0(DWORD dwCountToRead, VOID* pvBuffer);
MC_STATUS McReadPort2(DWORD dwCountToRead, VOID* pvBuffer);

MC_STATUS McBreakPort();

// Logging

void McMessageConnectionStatistics(MC_MESSAGE_LEVEL ml);

void McMessage(MC_MESSAGE_LEVEL ml, LPCSTR pszFormat, ...);

void McSuspendMessages();
void McResumeMessages();

LPCSTR McGetHexStr(LPCVOID lpBuffer, DWORD dwCount);
CHAR McGetHexChar(BYTE bDigit);
LPCSTR McGetTimeStr();

// Misc

void McDelay(DWORD dwMillisec);

MC_STATUS McBeginTimeCriticalOperation();
void McEndTimeCriticalOperation();

MC_STATUS McSetTimeCriticalThreadPriority();
void McRestoreThreadPriority();

///////////////////////////////////////////////////////////////////////////////////////

struct MC_CONNECTION_STATISTICS
{
	DWORD dwStartTickCount;
	DWORD dwEndTickCount;	
	DWORD dwBlockSentCount;
	DWORD dwBreakCount;
	DWORD dwBlockReceivedCount;
	DWORD dwResetCount;
	DWORD dwReestablishmentCount;
};

const DWORD MC_REQUEST_ATTEMPT_COUNT = 2;	// Number of attempts to execute a request
const DWORD MC_MAX_MESSAGE_SIZE = 0x400;

bool g_bConstructed = false;

const VOID* g_pvContext;
MC_MESSAGE_HANDLER g_mh;
MC_MESSAGE_LEVEL g_ml;

bool g_bConnected;

HANDLE g_hPort;
DCB g_dcbOld;
COMMTIMEOUTS g_commtimeoutsOld;
DWORD g_dwLastConnectionBreak;
double g_dTimerResolution;
LARGE_INTEGER g_liStartCount;
MC_CONNECTION_STATISTICS g_cs;

// Incoming blocks are of dynamic size. We 
// 1) don't want to alloc / free each time,
// 2) don't have to deal with more than one block at a time. 
// So we reuse the same memory buffer for blocks.

VAG_BLOCK* g_pvb;
BYTE g_bBlockSize;

BYTE g_bAddress;
BYTE g_bBlockIndex;

int g_iThreadPriorityOld;

LPSTR g_pszMessage;
DWORD g_dwMessageCapacity;
DWORD g_dwMessageLength;

bool g_bMessagesSuspended;

#ifdef MC_EMULATE_COMM
DWORD g_dwLastAccessTicks;
#endif

#ifdef MC_SWAP_RTS_DTR

#pragma warning (push)
#pragma warning (disable : 4005)

#define SETRTS 5
#define CLRRTS 6
#define SETDTR 3
#define CLRDTR 4

#pragma warning (pop)

#endif

MC_STATUS MC_API McConstruct(const VOID* pvContext, MC_MESSAGE_HANDLER mh)
{
	assert(! g_bConstructed);

	g_pvContext = pvContext;
	g_mh = mh;	
	g_ml = MC_ML_LOG_0;

	g_bConnected = false;
	
	g_hPort = INVALID_HANDLE_VALUE;
	
	g_dwLastConnectionBreak = 0;

	LARGE_INTEGER liFrequency;
	BOOL b = QueryPerformanceFrequency(&liFrequency);

	if (! b)
		return MC_S_ERROR;

	g_dTimerResolution = (double) 1000 / liFrequency.QuadPart;	

	QueryPerformanceCounter(&g_liStartCount);

	g_bBlockSize = 0;
	g_pvb = NULL;

	g_iThreadPriorityOld = THREAD_PRIORITY_ERROR_RETURN;

	g_pszMessage = NULL;
	g_dwMessageCapacity = 0;
	g_dwMessageLength = 0;

	g_bMessagesSuspended = false;


	g_bConstructed = true;

	return MC_S_OK;
}

void MC_API McDestruct()
{
	assert(! g_bConnected);
	assert(g_bConstructed);

	if (g_pszMessage != NULL)
		free(g_pszMessage);

	if (g_pvb != NULL)
		free(g_pvb);

	g_bConstructed = false;
}

void MC_API McSetMessagesLevel(MC_MESSAGE_LEVEL ml)
{
	assert(g_bConstructed);

	g_ml = ml;
}

MC_MESSAGE_LEVEL MC_API McGetMessagesLevel()
{
	assert(g_bConstructed);

	return g_ml;
}

VOID* MC_API McAllocate(SIZE_T size)
{
	if (! size)
		return NULL;

	return malloc(size);
}

VOID MC_API McFree(VOID* pv)
{
	if (pv != NULL)
		free(pv);
}

MC_STATUS MC_API McConnect(LPCSTR pszPortName, DWORD dwBaudRate, BYTE bAddress, LPSTR** pppszDescription)
{
	assert(g_bConstructed);
	assert(! g_bConnected);

	// Resetting the high resolution mark not to have huge values because of disconnected periods

	QueryPerformanceCounter(&g_liStartCount);

	if (! dwBaudRate)
		return MC_S_ERROR_IMPL;

	// Before we fully connect we should use a delay to cancel the handshake. Otherwise we may 
	// unintentionally send ECU some bytes.

	if (g_dwLastConnectionBreak)
	{
		// Ensuring the previous connection is cancelled with 2 * VAG_DISCONNECT_TIMEOUT timeout.

		DWORD dwDelta = GetTickCount() - g_dwLastConnectionBreak;

		if (dwDelta < 2 * VAG_DISCONNECT_TIMEOUT)
		{
			McDelay(2 * VAG_DISCONNECT_TIMEOUT - dwDelta);
		}

		g_dwLastConnectionBreak = 0;
	}

	g_cs.dwStartTickCount = GetTickCount();	
	g_cs.dwBlockSentCount = 0;
	g_cs.dwBreakCount = 0;
	g_cs.dwBlockReceivedCount = 0;
	g_cs.dwResetCount = 0;
	g_cs.dwReestablishmentCount = 0;

	// Opening comm port with the hinted baud rate

	MC_STATUS status = McOpenPort(pszPortName, dwBaudRate);

	if (status != MC_S_OK)
		return status;	

#ifndef MC_EMULATE_COMM

	McMessage(MC_ML_LOG_0, "Connecting to ECU expecting %u baud rate\n", (unsigned) dwBaudRate);

	status = McBeginTimeCriticalOperation();

	if (status != MC_S_OK)
	{
		McClosePort();

		return status;
	}

	DWORD dwFailCount = 0;

	CONNECT:

	// Trying to setup connection with ECU

	status = McSetupConnection(bAddress);

	if (status != MC_S_OK)
	{
		g_dwLastConnectionBreak = GetTickCount();
		McEndTimeCriticalOperation();
		McClosePort();

		return status;
	}

	// ECU sends us VAG_BTI_INFO_RES blocks automatically

	status = McReceiveBlockInfoRes(pppszDescription);

	if (status != MC_S_OK)
	{
		if (status == MC_S_ERROR_CONNECT_ECU && ++ dwFailCount < MC_REQUEST_ATTEMPT_COUNT)
		{
			McMessage(MC_ML_ERROR, "\nReestablishing connection\n\n");

			McDelay(2 * VAG_DISCONNECT_TIMEOUT);
			McMessage(MC_ML_LOG_0, "Connecting to ECU\n");

			goto CONNECT;
		}

		g_dwLastConnectionBreak = GetTickCount();
		McEndTimeCriticalOperation();
		McClosePort();

		return status;
	}

	McEndTimeCriticalOperation();

	g_cs.dwReestablishmentCount += dwFailCount;

#else

	McDelay(1000);

	if (dwBaudRate != 4800)
	{
		g_dwLastConnectionBreak = GetTickCount();
		McClosePort();

		return MC_S_ERROR_CONNECT;
	}

	LPSTR* ppszDescription = (LPSTR*) malloc(4 * sizeof(LPSTR));	
	ppszDescription[0] = strdup("443907311   ");
	ppszDescription[1] = strdup("   MOTOR    ");
	ppszDescription[2] = strdup("PMC  1");
	ppszDescription[3] = NULL;

	*pppszDescription = ppszDescription;

	g_dwLastAccessTicks = GetTickCount();

#endif

	g_bConnected = true;
	
	return MC_S_OK;
}

void MC_API McDisconnect()
{
	assert(g_bConnected);	

#ifndef MC_EMULATE_COMM

	McMessage(MC_ML_LOG_0, "\nDisconnecting from ECU\n");

	MC_STATUS status = McBeginTimeCriticalOperation();

	if (status == MC_S_OK)
	{
		// Sending VAG_BTI_COMM_END block

		VAG_BLOCK_COMM_END* pvbce = (VAG_BLOCK_COMM_END*) McAllocBlock(sizeof(VAG_BLOCK_COMM_END));
		pvbce->blockheader.bBlockIndex = ++ g_bBlockIndex;
		pvbce->blockheader.bBlockTypeId = VAG_BTI_COMM_END;

		status = McSendBlock(pvbce);

		if (status != MC_S_OK)
		{
			// Couldn't disconnect normally, then just wait.

			McMessage(MC_ML_LOG_1, "\nDisconnecting from ECU via timeout\n");

			g_dwLastConnectionBreak = GetTickCount();
		}

		McFreeBlock(pvbce);

		McEndTimeCriticalOperation();
	}
	else
	{
		McMessage(MC_ML_LOG_1, "\nDisconnecting from ECU via timeout\n");

		g_dwLastConnectionBreak = GetTickCount();
	}

	g_cs.dwEndTickCount = GetTickCount();

	McMessage(MC_ML_LOG_0, "\nDisconnected\n");

	McMessageConnectionStatistics(MC_ML_LOG_0);

#else

	McDelay(200);

#endif

	// Closing comm port

	McClosePort();

	g_bConnected = false;
}

DWORD MC_API McGetMaxIdleInterval()
{ 
	return VAG_DISCONNECT_TIMEOUT; 
}

MC_STATUS MC_API McIdle()
{
	assert(g_bConnected);

#ifndef MC_EMULATE_COMM

	MC_STATUS status = McBeginTimeCriticalOperation();

	if (status != MC_S_OK)
		return status;

	DWORD dwFailCount = 0;

	REQUEST:

	VAG_BLOCK_ACK* pvba = (VAG_BLOCK_ACK*) McAllocBlock(sizeof(VAG_BLOCK_ACK));
	pvba->blockheader.bBlockIndex = ++ g_bBlockIndex;
	pvba->blockheader.bBlockTypeId = VAG_BTI_ACK;

	status = McSendBlock(pvba);

	if (status != MC_S_OK)
	{
		McFreeBlock(pvba);

		if (status == MC_S_ERROR_CONNECT_ECU && ++ dwFailCount < MC_REQUEST_ATTEMPT_COUNT)
		{
			McMessage(MC_ML_ERROR, "\nReestablishing connection\n\n");

			status = McReconnect();

			if (status == MC_S_OK)
				goto REQUEST;
		}

		McEndTimeCriticalOperation();

		return status;
	}

	McFreeBlock(pvba);

	VAG_BLOCK* pvb;
	status = McReceiveBlock(&pvb);

	if (status != MC_S_OK)
	{
		if (status == MC_S_ERROR_CONNECT_ECU && ++ dwFailCount < MC_REQUEST_ATTEMPT_COUNT)
		{
			McMessage(MC_ML_ERROR, "\nReestablishing connection\n\n");

			status = McReconnect();

			if (status == MC_S_OK)
				goto REQUEST;
		}

		McEndTimeCriticalOperation();

		return status;
	}

	McEndTimeCriticalOperation();

	if (pvb->blockheader.bBlockIndex != ++ g_bBlockIndex)
	{
		McMessage(MC_ML_ERROR, "Bad block index: '%02X' vs '%02X'\n", (unsigned) 
			pvb->blockheader.bBlockIndex, (unsigned) g_bBlockIndex);

		g_bBlockIndex = pvb->blockheader.bBlockIndex;		// Synchronizing the block index
		McFreeBlock(pvb);

		return MC_S_ERROR_CONNECT_BLOCK_INDEX;
	}

	// Since pinging doesn't imply any action, VAG_BTI_ERROR is ok for it

	if (pvb->blockheader.bBlockTypeId != VAG_BTI_ACK && 
		pvb->blockheader.bBlockTypeId != VAG_BTI_ERROR)
	{
		McMessage(MC_ML_ERROR, "Bad block type id: '%02X' vs '%02X'\n", (unsigned) 
			pvb->blockheader.bBlockTypeId, (unsigned) VAG_BTI_ACK);

		McFreeBlock(pvb);

		return MC_S_ERROR_CONNECT;
	}

	McFreeBlock(pvb);

#else

	if (GetTickCount() - g_dwLastAccessTicks > VAG_DISCONNECT_TIMEOUT)
		return MC_S_ERROR_CONNECT_ECU;

	McDelay(100);

	g_dwLastAccessTicks = GetTickCount();

#endif

	return MC_S_OK;
}

MC_STATUS MC_API McGetDescriptions(LPSTR** pppszDescription)
{
	assert(g_bConnected);

#ifndef MC_EMULATE_COMM

	MC_STATUS status = McBeginTimeCriticalOperation();

	if (status != MC_S_OK)
		return status;

	DWORD dwFailCount = 0;

	REQUEST:

	VAG_BLOCK_INFO_REQ* pvbirq = (VAG_BLOCK_INFO_REQ*) McAllocBlock(sizeof(VAG_BLOCK_INFO_REQ));
	pvbirq->blockheader.bBlockIndex = ++ g_bBlockIndex;
	pvbirq->blockheader.bBlockTypeId = VAG_BTI_INFO_REQ;

	status = McSendBlock(pvbirq);

	if (status != MC_S_OK)
	{
		McFreeBlock(pvbirq);

		if (status == MC_S_ERROR_CONNECT_ECU && ++ dwFailCount < MC_REQUEST_ATTEMPT_COUNT)
		{
			McMessage(MC_ML_ERROR, "\nReestablishing connection\n\n");

			status = McReconnect();

			if (status == MC_S_OK)
				goto REQUEST;
		}

		McEndTimeCriticalOperation();		

		return status;
	}

	McFreeBlock(pvbirq);

	status = McReceiveBlockInfoRes(pppszDescription);

	if (status != MC_S_OK)
	{
		if (status == MC_S_ERROR_CONNECT_ECU && ++ dwFailCount < MC_REQUEST_ATTEMPT_COUNT)
		{
			McMessage(MC_ML_ERROR, "\nReestablishing connection\n\n");

			status = McReconnect();

			if (status == MC_S_OK)
				goto REQUEST;
		}

		McEndTimeCriticalOperation();

		return status;
	}

	McEndTimeCriticalOperation();

#else

	if (GetTickCount() - g_dwLastAccessTicks > VAG_DISCONNECT_TIMEOUT)
		return MC_S_ERROR_CONNECT_ECU;

	McDelay(500);

	LPSTR* ppszDescription = (LPSTR*) malloc(4 * sizeof(LPSTR));	
	ppszDescription[0] = strdup("443907311   ");
	ppszDescription[1] = strdup("   MOTOR    ");
	ppszDescription[2] = strdup("PMC  1");
	ppszDescription[3] = NULL;

	*pppszDescription = ppszDescription;

	g_dwLastAccessTicks = GetTickCount();

#endif

	return MC_S_OK;
}

MC_STATUS MC_API McGetFaults(DWORD* pdwCount, MC_FAULT** ppfault)
{
	assert(g_bConnected);

#ifndef MC_EMULATE_COMM

	DWORD dwMaxCount = 4;
	MC_FAULT* pfault = (MC_FAULT*) malloc(4 * sizeof(MC_FAULT));

	MC_STATUS status = McBeginTimeCriticalOperation();

	if (status != MC_S_OK)
	{
		free(pfault);

		return status;
	}

	DWORD dwFailCount = 0;

	REQUEST:

	VAG_BLOCK_DTC_REQ* pvbdrq = (VAG_BLOCK_DTC_REQ*) McAllocBlock(sizeof(VAG_BLOCK_DTC_REQ));
	pvbdrq->blockheader.bBlockIndex = ++ g_bBlockIndex;
	pvbdrq->blockheader.bBlockTypeId = VAG_BTI_DTC_REQ;

	status = McSendBlock(pvbdrq);

	if (status != MC_S_OK)
	{
		McFreeBlock(pvbdrq);

		if (status == MC_S_ERROR_CONNECT_ECU && ++ dwFailCount < MC_REQUEST_ATTEMPT_COUNT)
		{
			McMessage(MC_ML_ERROR, "\nReestablishing connection\n\n");

			status = McReconnect();

			if (status == MC_S_OK)
				goto REQUEST;
		}

		McEndTimeCriticalOperation();		
		free(pfault);

		return status;
	}

	McFreeBlock(pvbdrq);

	for (DWORD dwIndex = 0; ; ++ dwIndex)
	{
		VAG_BLOCK* pvb;
		MC_STATUS status = McReceiveBlock(&pvb);

		if (status != MC_S_OK)
		{
			if (status == MC_S_ERROR_CONNECT_ECU && ++ dwFailCount < MC_REQUEST_ATTEMPT_COUNT)
			{
				McMessage(MC_ML_ERROR, "\nReestablishing connection\n\n");

				status = McReconnect();

				if (status == MC_S_OK)
					goto REQUEST;
			}

			McEndTimeCriticalOperation();
			free(pfault);

			return status;
		}

		if (pvb->blockheader.bBlockIndex != ++ g_bBlockIndex)
		{
			McMessage(MC_ML_ERROR, "Bad block index: '%02X' vs '%02X'\n", (unsigned) 
				pvb->blockheader.bBlockIndex, (unsigned) g_bBlockIndex);

			g_bBlockIndex = pvb->blockheader.bBlockIndex;		// Synchronizing the block index			
			McFreeBlock(pvb);			
			McEndTimeCriticalOperation();
			free(pfault);

			return MC_S_ERROR_CONNECT_BLOCK_INDEX;
		}

		if (pvb->blockheader.bBlockTypeId == VAG_BTI_ERROR)
		{
			McFreeBlock(pvb);			
			McEndTimeCriticalOperation();
			free(pfault);

			return MC_S_ERROR_ECU;
		}

		if (pvb->blockheader.bBlockTypeId == VAG_BTI_ACK)
		{
			// VAG_BTI_ACK - no more DTCs

			McFreeBlock(pvb);
			
			break;
		}

		if (pvb->blockheader.bBlockTypeId != VAG_BTI_DTC_RES)
		{
			McMessage(MC_ML_ERROR, "Bad block type id: '%02X' vs '%02X'\n", (unsigned) 
				pvb->blockheader.bBlockTypeId, (unsigned) VAG_BTI_DTC_RES);

			McFreeBlock(pvb);			
			McEndTimeCriticalOperation();
			free(pfault);

			return MC_S_ERROR_CONNECT;
		}

		VAG_BLOCK_DTC_RES* pvbdrs = (VAG_BLOCK_DTC_RES*) pvb;

		if (dwIndex == dwMaxCount)
		{
			dwMaxCount += 4;
			pfault = (MC_FAULT*) realloc(pfault, dwMaxCount * sizeof(MC_FAULT));
		}

		MC_FAULT& rfault = pfault[dwIndex];
		rfault.bSporadic = pvbdrs->fSporadic != 0;
		rfault.wCode = pvbdrs->wCode >> 8 | pvbdrs->wCode << 8;
		rfault.bSubCode = pvbdrs->bSubCode;		

		McFreeBlock(pvbdrs);

		VAG_BLOCK_ACK* pvba = (VAG_BLOCK_ACK*) McAllocBlock(sizeof(VAG_BLOCK_ACK));
		pvba->blockheader.bBlockIndex = ++ g_bBlockIndex;
		pvba->blockheader.bBlockTypeId = VAG_BTI_ACK;

		status = McSendBlock(pvba);

		if (status != MC_S_OK)
		{
			McFreeBlock(pvba);

			if (status == MC_S_ERROR_CONNECT_ECU && ++ dwFailCount < MC_REQUEST_ATTEMPT_COUNT)
			{
				McMessage(MC_ML_ERROR, "\nReestablishing connection\n\n");

				status = McReconnect();

				if (status == MC_S_OK)
					goto REQUEST;
			}

			McEndTimeCriticalOperation();
			free(pfault);

			return status;
		}

		McFreeBlock(pvba);
	}

	McEndTimeCriticalOperation();

	*pdwCount = dwIndex;
	*ppfault = pfault;

#else

	if (GetTickCount() - g_dwLastAccessTicks > VAG_DISCONNECT_TIMEOUT)
		return MC_S_ERROR_CONNECT_ECU;

	McDelay(200);

	MC_FAULT* pfault = (MC_FAULT*) malloc(2 * sizeof(MC_FAULT));

	pfault[0].wCode = 515;
	pfault[0].bSubCode = 3;
	pfault[0].bSporadic = 0;
	pfault[1].wCode = 522;
	pfault[1].bSubCode = 30;
	pfault[1].bSporadic = 1;

	*pdwCount = 2;
	*ppfault = pfault;

	g_dwLastAccessTicks = GetTickCount();

#endif

	return MC_S_OK;
}

MC_STATUS MC_API McClearFaults()
{
	assert(g_bConnected);

#ifndef MC_EMULATE_COMM

	MC_STATUS status = McBeginTimeCriticalOperation();

	if (status != MC_S_OK)
		return status;

	DWORD dwFailCount = 0;

	REQUEST:

	VAG_BLOCK_DTC_CLR* pvbdc = (VAG_BLOCK_DTC_CLR*) McAllocBlock(sizeof(VAG_BLOCK_DTC_CLR));
	pvbdc->blockheader.bBlockIndex = ++ g_bBlockIndex;
	pvbdc->blockheader.bBlockTypeId = VAG_BTI_DTC_CLR;

	status = McSendBlock(pvbdc);

	if (status != MC_S_OK)
	{
		McFreeBlock(pvbdc);

		if (status == MC_S_ERROR_CONNECT_ECU && ++ dwFailCount < MC_REQUEST_ATTEMPT_COUNT)
		{
			McMessage(MC_ML_ERROR, "\nReestablishing connection\n\n");

			status = McReconnect();

			if (status == MC_S_OK)
				goto REQUEST;
		}

		McEndTimeCriticalOperation();		

		return status;
	}

	McFreeBlock(pvbdc);

	VAG_BLOCK* pvb;
	status = McReceiveBlock(&pvb);

	if (status != MC_S_OK)
	{
		if (status == MC_S_ERROR_CONNECT_ECU && ++ dwFailCount < MC_REQUEST_ATTEMPT_COUNT)
		{
			McMessage(MC_ML_ERROR, "\nReestablishing connection\n\n");

			status = McReconnect();

			if (status == MC_S_OK)
				goto REQUEST;
		}

		McEndTimeCriticalOperation();

		return status;
	}

	McEndTimeCriticalOperation();

	if (pvb->blockheader.bBlockIndex != ++ g_bBlockIndex)
	{
		McMessage(MC_ML_ERROR, "Bad block index: '%02X' vs '%02X'\n", (unsigned) 
			pvb->blockheader.bBlockIndex, (unsigned) g_bBlockIndex);

		g_bBlockIndex = pvb->blockheader.bBlockIndex;		// Synchronizing the block index
		McFreeBlock(pvb);

		return MC_S_ERROR_CONNECT_BLOCK_INDEX;
	}

	if (pvb->blockheader.bBlockTypeId == VAG_BTI_ERROR)
	{
		McFreeBlock(pvb);

		return MC_S_ERROR_ECU;
	}

	if (pvb->blockheader.bBlockTypeId != VAG_BTI_ACK)
	{
		McMessage(MC_ML_ERROR, "Bad block type id: '%02X' vs '%02X'\n", (unsigned) 
			pvb->blockheader.bBlockTypeId, (unsigned) VAG_BTI_ACK);

		McFreeBlock(pvb);

		return MC_S_ERROR_CONNECT;
	}

	McFreeBlock(pvb);

#else

	if (GetTickCount() - g_dwLastAccessTicks > VAG_DISCONNECT_TIMEOUT)
		return MC_S_ERROR_CONNECT_ECU;

	McDelay(200);

	g_dwLastAccessTicks = GetTickCount();

#endif

	return MC_S_OK;
}

MC_STATUS MC_API McGetBasicSettingsVar(DWORD* pdwCount, BYTE** ppbZone)
{
	assert(g_bConnected);

#ifndef MC_EMULATE_COMM

	MC_STATUS status = McBeginTimeCriticalOperation();

	if (status != MC_S_OK)
		return status;

	DWORD dwFailCount = 0;

	REQUEST:

	VAG_BLOCK_BS_REQ* pvbbrq = (VAG_BLOCK_BS_REQ*) McAllocBlock(sizeof(VAG_BLOCK_BS_REQ));
	pvbbrq->blockheader.bBlockIndex = ++ g_bBlockIndex;
	pvbbrq->blockheader.bBlockTypeId = VAG_BTI_BS_REQ;

	status = McSendBlock(pvbbrq);

	if (status != MC_S_OK)
	{
		McFreeBlock(pvbbrq);

		if (status == MC_S_ERROR_CONNECT_ECU && ++ dwFailCount < MC_REQUEST_ATTEMPT_COUNT)
		{
			McMessage(MC_ML_ERROR, "\nReestablishing connection\n\n");

			status = McReconnect();

			if (status == MC_S_OK)
				goto REQUEST;
		}

		McEndTimeCriticalOperation();

		return status;
	}

	McFreeBlock(pvbbrq);

	VAG_BLOCK* pvb;
	status = McReceiveBlock(&pvb);

	if (status != MC_S_OK)
	{
		if (status == MC_S_ERROR_CONNECT_ECU && ++ dwFailCount < MC_REQUEST_ATTEMPT_COUNT)
		{
			McMessage(MC_ML_ERROR, "\nReestablishing connection\n\n");

			status = McReconnect();

			if (status == MC_S_OK)
				goto REQUEST;
		}

		McEndTimeCriticalOperation();

		return status;
	}

	McEndTimeCriticalOperation();

	if (pvb->blockheader.bBlockIndex != ++ g_bBlockIndex)
	{
		McMessage(MC_ML_ERROR, "Bad block index: '%02X' vs '%02X'\n", (unsigned) 
			pvb->blockheader.bBlockIndex, (unsigned) g_bBlockIndex);

		g_bBlockIndex = pvb->blockheader.bBlockIndex;		// Synchronizing the block index
		McFreeBlock(pvb);

		return MC_S_ERROR_CONNECT_BLOCK_INDEX;
	}

	if (pvb->blockheader.bBlockTypeId == VAG_BTI_ERROR)
	{
		McFreeBlock(pvb);

		return MC_S_ERROR_ECU;
	}

	if (pvb->blockheader.bBlockTypeId != VAG_BTI_BS_RES)
	{
		McMessage(MC_ML_ERROR, "Bad block type id: '%02X' vs '%02X'\n", (unsigned) 
			pvb->blockheader.bBlockTypeId, (unsigned) VAG_BTI_BS_RES);

		McFreeBlock(pvb);

		return MC_S_ERROR_CONNECT;
	}

	VAG_BLOCK_BS_RES* pvbbrs = (VAG_BLOCK_BS_RES*) pvb;

	BYTE bCount = pvbbrs->blockheader.bBlockSize - sizeof(VAG_BLOCK_HEADER);

	*pdwCount = bCount;

	if (bCount)
	{
		BYTE* pbZone = (BYTE*) malloc(bCount);
		
		memcpy(pbZone, pvbbrs->abZone, bCount);		// nothing to convert

		*ppbZone = pbZone;
	}
	else
		*ppbZone = NULL;

	McFreeBlock(pvbbrs);

#else

	if (GetTickCount() - g_dwLastAccessTicks > VAG_DISCONNECT_TIMEOUT)
		return MC_S_ERROR_CONNECT_ECU;

	McDelay(400);

	BYTE* pbZone = (BYTE*) malloc(10);

	pbZone[0] = 19;
	pbZone[1] = 33;
	pbZone[2] = 34;
	pbZone[3] = 85;
	pbZone[4] = 128;
	pbZone[5] = 252;
	pbZone[6] = 254;
	pbZone[7] = 5;
	pbZone[8] = 174;
	pbZone[9] = 2;

	*pdwCount = 10;
	*ppbZone = pbZone;

	g_dwLastAccessTicks = GetTickCount();

#endif

	return MC_S_OK;
}

MC_STATUS MC_API McGetBasicSettings(DWORD dwCount, BYTE* pbZone)
{
	assert(g_bConnected);

#ifndef MC_EMULATE_COMM

	MC_STATUS status = McBeginTimeCriticalOperation();

	if (status != MC_S_OK)
		return status;

	DWORD dwFailCount = 0;

	REQUEST:

	VAG_BLOCK_BS_REQ* pvbbrq = (VAG_BLOCK_BS_REQ*) McAllocBlock(sizeof(VAG_BLOCK_BS_REQ));
	pvbbrq->blockheader.bBlockIndex = ++ g_bBlockIndex;
	pvbbrq->blockheader.bBlockTypeId = VAG_BTI_BS_REQ;

	status = McSendBlock(pvbbrq);

	if (status != MC_S_OK)
	{
		McFreeBlock(pvbbrq);

		if (status == MC_S_ERROR_CONNECT_ECU && ++ dwFailCount < MC_REQUEST_ATTEMPT_COUNT)
		{
			McMessage(MC_ML_ERROR, "\nReestablishing connection\n\n");

			status = McReconnect();

			if (status == MC_S_OK)
				goto REQUEST;
		}

		McEndTimeCriticalOperation();

		return status;
	}

	McFreeBlock(pvbbrq);

	VAG_BLOCK* pvb;
	status = McReceiveBlock(&pvb);

	if (status != MC_S_OK)
	{
		if (status == MC_S_ERROR_CONNECT_ECU && ++ dwFailCount < MC_REQUEST_ATTEMPT_COUNT)
		{
			McMessage(MC_ML_ERROR, "\nReestablishing connection\n\n");

			status = McReconnect();

			if (status == MC_S_OK)
				goto REQUEST;
		}

		McEndTimeCriticalOperation();

		return status;
	}

	McEndTimeCriticalOperation();

	if (pvb->blockheader.bBlockIndex != ++ g_bBlockIndex)
	{
		McMessage(MC_ML_ERROR, "Bad block index: '%02X' vs '%02X'\n", (unsigned) 
			pvb->blockheader.bBlockIndex, (unsigned) g_bBlockIndex);

		g_bBlockIndex = pvb->blockheader.bBlockIndex;		// Synchronizing the block index
		McFreeBlock(pvb);

		return MC_S_ERROR_CONNECT_BLOCK_INDEX;
	}

	if (pvb->blockheader.bBlockTypeId == VAG_BTI_ERROR)
	{
		McFreeBlock(pvb);

		return MC_S_ERROR_ECU;
	}

	if (pvb->blockheader.bBlockTypeId != VAG_BTI_BS_RES)
	{
		McMessage(MC_ML_ERROR, "Bad block type id: '%02X' vs '%02X'\n", (unsigned) 
			pvb->blockheader.bBlockTypeId, (unsigned) VAG_BTI_BS_RES);

		McFreeBlock(pvb);

		return MC_S_ERROR_CONNECT;
	}

	VAG_BLOCK_BS_RES* pvbbrs = (VAG_BLOCK_BS_RES*) pvb;

	BYTE bCount = pvbbrs->blockheader.bBlockSize - sizeof(VAG_BLOCK_HEADER);

	if (bCount != dwCount)
	{
		McMessage(MC_ML_ERROR, "Bad zone count: %u vs %u\n", (unsigned) bCount, 
			(unsigned) dwCount);

		McFreeBlock(pvbbrs);

		return MC_S_ERROR;
	}

	memcpy(pbZone, pvbbrs->abZone, dwCount);		// nothing to convert

	McFreeBlock(pvbbrs);

#else

	if (GetTickCount() - g_dwLastAccessTicks > VAG_DISCONNECT_TIMEOUT)
		return MC_S_ERROR_CONNECT_ECU;

	McDelay(400);

	if (dwCount != 10)
	{
		McMessage(MC_ML_ERROR, "Bad zone count: %u vs %u\n", (unsigned) 10, 
			(unsigned) dwCount);

		return MC_S_ERROR;
	}

	static double time = 0;

	pbZone[4] = (int) (128 + 40 * sin(time));
	time += 0.5;

	pbZone[0] = 19;
	pbZone[1] = 33;
	pbZone[2] = 34;
	pbZone[3] = 85;
	pbZone[5] = 252;
	pbZone[6] = 254;
	pbZone[7] = 5;
	pbZone[8] = 174;
	pbZone[9] = 2;

	g_dwLastAccessTicks = GetTickCount();

#endif

	return MC_S_OK;
}

MC_STATUS MC_API McTestActuator(WORD* pdwActuatorCode)
{
	assert(g_bConnected);

#ifndef MC_EMULATE_COMM

	MC_STATUS status = McBeginTimeCriticalOperation();

	if (status != MC_S_OK)
		return status;

	DWORD dwFailCount = 0;

	REQUEST:

	VAG_BLOCK_ACT_REQ* pvbarq = (VAG_BLOCK_ACT_REQ*) McAllocBlock(sizeof(VAG_BLOCK_ACT_REQ));
	pvbarq->blockheader.bBlockIndex = ++ g_bBlockIndex;
	pvbarq->blockheader.bBlockTypeId = VAG_BTI_ACT_REQ;
	pvbarq->bUnknown = 0;

	status = McSendBlock(pvbarq);

	if (status != MC_S_OK)
	{
		McFreeBlock(pvbarq);

		if (status == MC_S_ERROR_CONNECT_ECU && ++ dwFailCount < MC_REQUEST_ATTEMPT_COUNT)
		{
			McMessage(MC_ML_ERROR, "\nReestablishing connection\n\n");

			status = McReconnect();

			if (status == MC_S_OK)
				goto REQUEST;
		}

		McEndTimeCriticalOperation();		

		return status;
	}

	McFreeBlock(pvbarq);

	VAG_BLOCK* pvb;
	status = McReceiveBlock(&pvb);

	if (status != MC_S_OK)
	{
		if (status == MC_S_ERROR_CONNECT_ECU && ++ dwFailCount < MC_REQUEST_ATTEMPT_COUNT)
		{
			McMessage(MC_ML_ERROR, "\nReestablishing connection\n\n");

			status = McReconnect();

			if (status == MC_S_OK)
				goto REQUEST;
		}

		McEndTimeCriticalOperation();

		return status;
	}

	McEndTimeCriticalOperation();

	if (pvb->blockheader.bBlockIndex != ++ g_bBlockIndex)
	{
		McMessage(MC_ML_ERROR, "Bad block index: '%02X' vs '%02X'\n", (unsigned) 
			pvb->blockheader.bBlockIndex, (unsigned) g_bBlockIndex);

		g_bBlockIndex = pvb->blockheader.bBlockIndex;		// Synchronizing the block index
		McFreeBlock(pvb);

		return MC_S_ERROR_CONNECT_BLOCK_INDEX;
	}

	if (pvb->blockheader.bBlockTypeId == VAG_BTI_ERROR)
	{
		McFreeBlock(pvb);

		return MC_S_ERROR_ECU;
	}

	if (pvb->blockheader.bBlockTypeId == VAG_BTI_ACK)
	{
		// No more actuators to test

		McFreeBlock(pvb);

		*pdwActuatorCode = 0;

		return MC_S_OK;
	}

	if (pvb->blockheader.bBlockTypeId != VAG_BTI_ACT_RES)
	{
		McMessage(MC_ML_ERROR, "Bad block type id: '%02X' vs '%02X'\n", (unsigned) 
			pvb->blockheader.bBlockTypeId, (unsigned) VAG_BTI_ACT_RES);

		McFreeBlock(pvb);

		return MC_S_ERROR_CONNECT;
	}

	VAG_BLOCK_ACT_RES* pvbars = (VAG_BLOCK_ACT_RES*) pvb;

	*pdwActuatorCode = pvbars->wActuatorCode >> 8 | pvbars->wActuatorCode << 8;

	McFreeBlock(pvbars);

#else

	static WORD wActuatorCode = 0;

	if (GetTickCount() - g_dwLastAccessTicks > VAG_DISCONNECT_TIMEOUT)
		return MC_S_ERROR_CONNECT_ECU;

	if (wActuatorCode < 3)
	{
		++ wActuatorCode;
	}
	else
		wActuatorCode = 0;

	*pdwActuatorCode = wActuatorCode;

	g_dwLastAccessTicks = GetTickCount();

#endif

	return MC_S_OK;
}

MC_STATUS McOpenPort(LPCSTR pszPortName, DWORD dwBaudRate)
{
	assert(g_hPort == INVALID_HANDLE_VALUE);

	// Opening comm port

	g_hPort = CreateFile(
		pszPortName, 
		GENERIC_READ | GENERIC_WRITE,
		0,
		NULL,
		OPEN_EXISTING,
		0,
		NULL);

	if (g_hPort == INVALID_HANDLE_VALUE)
		return MC_S_ERROR_CONNECT_PORT;

	// Getting comm port setup

	BOOL b = GetCommState(g_hPort, &g_dcbOld);

	if (! b)
	{
		CloseHandle(g_hPort);
		g_hPort = INVALID_HANDLE_VALUE;

		return MC_S_ERROR_CONNECT_PORT;
	}

	b = GetCommTimeouts(g_hPort, &g_commtimeoutsOld);

	if (! b)
	{
		CloseHandle(g_hPort);
		g_hPort = INVALID_HANDLE_VALUE;

		return MC_S_ERROR_CONNECT_PORT;
	}

	// Setting up comm port

	DCB dcb =
	{
		sizeof(DCB),
		dwBaudRate,
		TRUE,
		FALSE,					// No parity checking on reception
		FALSE,					// No CTS checking on transmission
		FALSE,					// No DSR checking on transmission
		DTR_CONTROL_DISABLE,	// Some adapters may need the negative voltage
		FALSE,					// No DSR checking on reception
		TRUE,
		FALSE,					// No Xoff/Xon checking on transmission
		FALSE,					// No Xoff/Xon sending on reception
		FALSE,					// No replacement of bytes with bad parity
		FALSE,					// No zero bytes removal
		RTS_CONTROL_DISABLE,	// !
		FALSE,					// No latchup on errors
		0,
		0,
		0,
		0,
		8,						// 8 data bits per symbol
		NOPARITY,				// No parity generation on transmission
		ONESTOPBIT,				// 1 stop bit
		'\0',
		'\0',
		'\0',
		'\0',					// End of data char
		'\0',					// Event char
		0
	};

	b = SetCommState(g_hPort, &dcb);

	if (! b)
	{
		CloseHandle(g_hPort);
		g_hPort = INVALID_HANDLE_VALUE;

		return MC_S_ERROR_CONNECT_PORT;
	}

	b = SetCommMask(g_hPort, 0);

	if (! b)
	{
		SetCommTimeouts(g_hPort, &g_commtimeoutsOld);
		SetCommState(g_hPort, &g_dcbOld);
		CloseHandle(g_hPort);
		g_hPort = INVALID_HANDLE_VALUE;

		return MC_S_ERROR_CONNECT_PORT;
	}

	// Flushing buffers and setting the new ones
    
	b = SetupComm(g_hPort, 0x100, 0x100);

	if (! b)
	{
		SetCommTimeouts(g_hPort, &g_commtimeoutsOld);
		SetCommState(g_hPort, &g_dcbOld);
		CloseHandle(g_hPort);
		g_hPort = INVALID_HANDLE_VALUE;

		return MC_S_ERROR_CONNECT_PORT;
	}

	// TX goes 1
	// No need for PurgeComm() ?

	return MC_S_OK;
}

void McClosePort()
{
	assert(g_hPort != INVALID_HANDLE_VALUE);

	// Restoring comm port setup

	SetCommState(g_hPort, &g_dcbOld);

	SetCommTimeouts(g_hPort, &g_commtimeoutsOld);

	// Closing comm port

	CloseHandle(g_hPort);

	g_hPort = INVALID_HANDLE_VALUE;
}

// This function is factored out because of the reuse only

MC_STATUS McSetupConnection(BYTE bAddress)
{
	// The most weird part of VAG communication is the connection setup. Especially
	// if you cannot process VAG_SIG_BAUD_RATE_SYNCH and determine ECU baud rate.
	// So think TWICE before changing or moving any line here.

	MC_STATUS status = McSetPortTimeout(VAG_DISCONNECT_TIMEOUT, 0);

	if (status != MC_S_OK)
		return status;

	// Sending ECU address to K and L lines with VAG_CONNECT_BAUD_RATE / 7 bits /
	// odd parity / 1 stop bit.

	// We don't send the ECU address natively, setting up VAG_CONNECT_BAUD_RATE because
	// 1) we may not have time to switch to the hinted baud rate after the transmission
	// 2) we have to send the address to L line too	

	// Note, any 1 -> 0 front at VAG_CONNECT_BAUD_RATE including the start bit will echo 
	// 0x00 byte at the much higher hinted baud rate.

	const DWORD VAG_CONNECT_BIT_LENGTH = 1000 / VAG_CONNECT_BAUD_RATE;

	McDelay(VAG_CONNECT_BIT_LENGTH);

	// Sending start bit (0)

	BOOL b = EscapeCommFunction(g_hPort, SETBREAK);

	if (! b)
		return MC_S_ERROR_CONNECT_PORT;

	b = EscapeCommFunction(g_hPort, SETRTS);

	if (! b)
		return MC_S_ERROR_CONNECT_PORT;

	BYTE bData = bAddress;	
	BOOL bParity = TRUE;
	DWORD dwBreak, dwRts;

	for (int i = 0; i < 7; ++ i)
	{
		McDelay(VAG_CONNECT_BIT_LENGTH);

		// Sending address bit

		if (bData & 0x01)
		{
			dwBreak = CLRBREAK;
			dwRts = CLRRTS;
			bParity = ! bParity;
		}
		else
		{
			dwBreak = SETBREAK;
			dwRts = SETRTS;
		}

		b = EscapeCommFunction(g_hPort, dwBreak);

		if (! b)
			return MC_S_ERROR_CONNECT_PORT;

		b = EscapeCommFunction(g_hPort, dwRts);

		if (! b)
			return MC_S_ERROR_CONNECT_PORT;

		bData >>= 1;
	}

	McDelay(VAG_CONNECT_BIT_LENGTH);

	// Sending parity bit	

	if (bParity)
	{
		dwBreak = CLRBREAK;
		dwRts = CLRRTS;
	}
	else
	{
		dwBreak = SETBREAK;
		dwRts = SETRTS;
	}

	b = EscapeCommFunction(g_hPort, dwBreak);

	if (! b)
		return MC_S_ERROR_CONNECT_PORT;

	b = EscapeCommFunction(g_hPort, dwRts);

	if (! b)
		return MC_S_ERROR_CONNECT_PORT;

	McDelay(VAG_CONNECT_BIT_LENGTH);

	// Processing echos

	DWORD dwErrors;
	COMSTAT comstat;
	b = ClearCommError(g_hPort, &dwErrors, &comstat);

	if (! b)
		return MC_S_ERROR_CONNECT_PORT;

	if (! comstat.cbInQue)
	{
		// No adapter echo. At least the start bit should have caused an echo byte. 

		McMessage(MC_ML_ERROR, "No adapter echo\n");

		return MC_S_ERROR_CONNECT_ADAPTER;
	}

	b = PurgeComm(g_hPort, PURGE_RXCLEAR);

	if (! b)
		return MC_S_ERROR_CONNECT_PORT;

	// Sending stop bit

	b = EscapeCommFunction(g_hPort, CLRBREAK);

	if (! b)
		return MC_S_ERROR_CONNECT_PORT;

	b = EscapeCommFunction(g_hPort, CLRRTS);

	if (! b)
		return MC_S_ERROR_CONNECT_PORT;

	// We read VAG_SIG_BAUD_RATE_SYNCH and VAG_SIG_MAGIC separately to enable
	// the timeout for each.

	// Reading and checking VAG_SIG_BAUD_RATE_SYNCH

	BYTE bBaudRateSynch;	
	status = McReadPort0(1, &bBaudRateSynch);

	if (status != MC_S_OK)
		return status;

	if (bBaudRateSynch != VAG_SIG_BAUD_RATE_SYNCH)
	{
		McMessage(MC_ML_ERROR, "Bad Baud Rate Synch Byte: '%02X' vs '%02X'\n", (unsigned) 
			bBaudRateSynch, (unsigned) VAG_SIG_BAUD_RATE_SYNCH);

		return MC_S_ERROR_CONNECT;	// MC_S_ERROR_CONNECT_BAUD_RATE ?? Not yet.
	}

	McMessage(MC_ML_LOG_1, "Baud Rate Synch Byte is Ok\n");

	// The hinted baud rate matches the ECU baud rate.

	// Reading and checking VAG_SIG_MAGIC

	WORD wMagic;
	status = McReadPort0(2, &wMagic);

	if (status != MC_S_OK)
		return status;

	if (wMagic != VAG_SIG_MAGIC)
	{
		McMessage(MC_ML_ERROR, "Bad Magic Bytes: '%04X' vs '%04X'\n", (unsigned) wMagic,
			(unsigned) VAG_SIG_MAGIC);

		return MC_S_ERROR_CONNECT;
	}

	McMessage(MC_ML_LOG_1, "Magic Bytes are Ok\n");

	// Sending VAG_SIG_MAGIC_ACK at ECU baud rate

	BYTE bAck = VAG_SIG_MAGIC_ACK;
	status = McWritePort1(1, &bAck);

	if (status != MC_S_OK)
		return status;

	McMessage(MC_ML_LOG_0, "Connected successfully\n");

	g_bAddress = bAddress;
	g_bBlockIndex = 0;

	return MC_S_OK;
}

// This function is factored out because of the reuse only

MC_STATUS McReceiveBlockInfoRes(LPSTR** pppszDescription)
{
	DWORD dwMaxCount = 8;
	LPSTR* ppszDescription = (LPSTR*) malloc(8 * sizeof(LPSTR));

	for (int i = 0; ; ++ i)
	{
		if (i == dwMaxCount)
		{
			dwMaxCount += 8;
			ppszDescription = (LPSTR*) realloc(ppszDescription, dwMaxCount * sizeof(LPSTR));
		}		

		VAG_BLOCK* pvb;
		MC_STATUS status = McReceiveBlock(&pvb);

		if (status != MC_S_OK)
		{
			free(ppszDescription);

			return status;
		}

		if (pvb->blockheader.bBlockIndex != ++ g_bBlockIndex)
		{
			McMessage(MC_ML_ERROR, "Bad block index: '%02X' vs '%02X'\n", (unsigned) 
				pvb->blockheader.bBlockIndex, (unsigned) g_bBlockIndex);

			McFreeBlock(pvb);
			free(ppszDescription);

			return MC_S_ERROR_CONNECT_BLOCK_INDEX;
		}

		if (pvb->blockheader.bBlockTypeId == VAG_BTI_ERROR)
		{
			McFreeBlock(pvb);
			free(ppszDescription);

			return MC_S_ERROR_ECU;
		}

		if (pvb->blockheader.bBlockTypeId == VAG_BTI_ACK)
		{
			// VAG_BTI_ACK - no more descriptions

			McFreeBlock(pvb);

			ppszDescription[i] = NULL;
			
			break;
		}

		if (pvb->blockheader.bBlockTypeId != VAG_BTI_INFO_RES)
		{
			McMessage(MC_ML_ERROR, "Bad block type id: '%02X' vs '%02X'\n", (unsigned) 
				pvb->blockheader.bBlockTypeId, (unsigned) VAG_BTI_INFO_RES);

			McFreeBlock(pvb);
			free(ppszDescription);

			return MC_S_ERROR_CONNECT;
		}

		VAG_BLOCK_INFO_RES* pvbirs = (VAG_BLOCK_INFO_RES*) pvb;

		DWORD dwLength = pvbirs->blockheader.bBlockSize - sizeof(VAG_BLOCK_HEADER);
		LPSTR pszDescription = (LPSTR) malloc(dwLength + 1);
		memcpy(pszDescription, pvbirs->acInfo, dwLength);
		pszDescription[dwLength] = '\0';

		ppszDescription[i] = pszDescription;

		McFreeBlock(pvbirs);

		VAG_BLOCK_ACK* pvba = (VAG_BLOCK_ACK*) McAllocBlock(sizeof(VAG_BLOCK_ACK));
		pvba->blockheader.bBlockIndex = ++ g_bBlockIndex;
		pvba->blockheader.bBlockTypeId = VAG_BTI_ACK;

		status = McSendBlock(pvba);

		if (status != MC_S_OK)
		{
			McFreeBlock(pvba);
			free(ppszDescription);

			return status;
		}

		McFreeBlock(pvba);
	}

	*pppszDescription = ppszDescription;

	return MC_S_OK;
}

// Sends a block to ECU followed by VAG_SIG_BLOCK_END.
//
// How the block is allocated doesn't matter

MC_STATUS McSendBlock(const VAG_BLOCK* pvb)
{
	const DWORD MC_SEND_ATTEMPT_COUNT = 3;		// Number of attempts to send the block

	if (g_ml >= MC_ML_LOG_1)
		McMessage(MC_ML_LOG_1, "\n---- ECU <<< MonoComm ----\n\n");

	DWORD dwFailCount = 0;
	
	SEND:

	// Sending the block

	MC_STATUS status = McSetPortTimeout(0, VAG_BREAK_TIMEOUT);

	if (status != MC_S_OK)
		return status;

	status = McWritePort2(pvb->blockheader.bBlockSize, pvb);

	if (status != MC_S_OK)
	{
		if ((status == MC_S_ERROR_CONNECT_ECU || status == MC_S_ERROR_CONNECT_ECU_ECHO) &&
			++ dwFailCount < MC_SEND_ATTEMPT_COUNT)
		{
			// Note, no or bad echo - doesn't matter - noise may mask the whole symbol

			McMessage(MC_ML_ERROR, "\nResending block\n\n");

			// Letting ECU know the block will be repeated
		
			status = McBreakPort();

			if (status == MC_S_OK)
			{
				++ g_cs.dwBreakCount;

				goto SEND;
			}

			if (status != MC_S_ERROR_CONNECT)
			{
				// Error in error recovering. Little sense to go on.

				return status;
			}

			// Sometimes for some reason ReadFile() doesn't wait for the 
			// specified timeout and finishes much earlier and hence 
			// returns less bytes than requested and McWritePort2() fails
			// false and McBreakPort() fails false.
			//
			// <<< [1] '11' @ 26950.9 ms - Ok [1] @ 26951.3 ms
			// >>> [2] @ 26951.8 ms - Ok [1] '11' @ 26963.0 ms	// 11 ms ?!
			// No ECU echo
			//
			// Resending block
			//
			// Breaking @ 26965.3 ms - Failed [1] 'EE' @ 27036.9 ms
			//
			// This second attempt is a simple workround for that bad 
			// ReadFile() behaviour, it should NOT be here, it's 
			// incorrect actually.

			status = McBreakPort();

			if (status == MC_S_OK)
			{
				++ g_cs.dwBreakCount;

				goto SEND;
			}

			// Error in error recovering. Little sense to go on.

			return status;
		}

		return status;
	}

	// Sending the block end to comfirm the last byte is ok

	if (g_ml >= MC_ML_LOG_2)
		McMessage(MC_ML_LOG_2, "\n");

	BYTE bBlockEnd = VAG_SIG_BLOCK_END;

#ifdef MC_EMULATE_COMM_ERRORS

	if (pvb->blockheader.bBlockIndex == 0x18 || pvb->blockheader.bBlockIndex == 0x19)
	{
		McMessage(MC_ML_ERROR, "Simulating end of block write error\n");

		bBlockEnd = (BYTE) ~bBlockEnd;
	}

#endif

	// ECU doesn't echo the end of the block. It simply fails to execute the request if 
	// the end of the request block is incorrect.

	status = McWritePort1(1, &bBlockEnd);

	if (status != MC_S_OK)
		return status;

	++ g_cs.dwBlockSentCount;

	return MC_S_OK;
}

// Receieves a block from ECU followed by VAG_SIG_BLOCK_END.
//
// If returned status is MC_S_OK the returned block must be freed with 
// McFreeBlock() before the next call to McReceiveBlock() or McAllocBlock().

MC_STATUS McReceiveBlock(VAG_BLOCK** ppvb)
{
	if (g_ml >= MC_ML_LOG_1)
		McMessage(MC_ML_LOG_1, "\n---- ECU >>> MonoComm ----\n\n");

	RECEIVE:

	// Reading block size first

	MC_STATUS status = McSetPortTimeout(VAG_DISCONNECT_TIMEOUT, 0);

	if (status != MC_S_OK)
		return status;

	BYTE bSize;
	status = McReadPort2(1, &bSize);

	if (status != MC_S_OK)
		return status;

	// Validating block size

	if (bSize < sizeof(VAG_BLOCK) || bSize > 0x80)
	{
		// Letting ECU recover for the case of noise

		if (g_ml >= MC_ML_LOG_2)
			McMessage(MC_ML_LOG_2, "\n");

		status = McSetPortTimeout(0, VAG_BREAK_TIMEOUT);

		if (status != MC_S_OK)
			return status;

		BYTE bData;
		status = McReadPort2(1, &bData);

		if (status == MC_S_OK)
		{
			McMessage(MC_ML_ERROR, "Bad block size: '%02X'\n", (unsigned) bSize);

			// No sense to go on with such a size

			return MC_S_ERROR_CONNECT;
		}

		if (status == MC_S_ERROR_CONNECT_ECU)
		{
			McMessage(MC_ML_ERROR, "\nResetting block\n\n");

			++ g_cs.dwResetCount;

			goto RECEIVE;
		}

		return status;
	}

	VAG_BLOCK* pvb = McAllocBlock(bSize);

	// Reading the rest of the block

	if (g_ml >= MC_ML_LOG_2)
		McMessage(MC_ML_LOG_2, "\n");

	status = McSetPortTimeout(0, VAG_BREAK_TIMEOUT);

	if (status != MC_S_OK)
	{
		McFreeBlock(pvb);

		return status;
	}

	status = McReadPort2(bSize - 1, (BYTE*) pvb + 1);

	if (status != MC_S_OK)
	{
		McFreeBlock(pvb);

		if (status == MC_S_ERROR_CONNECT_ECU)
		{
			// Broken, the block will be repeated

			McMessage(MC_ML_ERROR, "\nResetting block\n\n");

			++ g_cs.dwResetCount;

			goto RECEIVE;
		}

		return status;
	}

	// Reading the block end to see if the last byte is ok

	if (g_ml >= MC_ML_LOG_2)
		McMessage(MC_ML_LOG_2, "\n");

	BYTE bBlockEnd;
	status = McReadPort0(1, &bBlockEnd);

	if (status != MC_S_OK)
	{
		McFreeBlock(pvb);		

		if (status == MC_S_ERROR_CONNECT_ECU)
		{
			// Broken, the block will be repeated

			McMessage(MC_ML_ERROR, "\nResetting block\n\n");

			++ g_cs.dwResetCount;

			goto RECEIVE;
		}

		return status;
	}

	if (bBlockEnd != VAG_SIG_BLOCK_END)
	{
		McMessage(MC_ML_ERROR, "Bad end of block byte: '%02X' vs '%02X'\n", 
			(unsigned) bBlockEnd, (unsigned) VAG_SIG_BLOCK_END);
	}

	++ g_cs.dwBlockReceivedCount;

	*ppvb = pvb;

	return MC_S_OK;
}

// Reestablishes connection with ECU.
//
// Should only be called if ECU request timed out - failed with MC_S_ERROR_CONNECT_ECU.

MC_STATUS McReconnect()
{
	assert(g_bMessagesSuspended && g_iThreadPriorityOld != THREAD_PRIORITY_ERROR_RETURN);	

	McMessage(MC_ML_LOG_0, "Connecting to ECU\n");

	// There's an outside chance we get here not because of a disconnect but because of 
	// noise that masks 3 ECU echos. So breaking the existing connection first

	McDelay(2 * VAG_DISCONNECT_TIMEOUT);

	// Setting up new connection using the current baud rate and ECU address.

	MC_STATUS status = McSetupConnection(g_bAddress);

	if (status != MC_S_OK)
	{
		// Note, we cannot use g_dwLastConnectionBreak here, because the next operation
		// may be other than McConnect()

		McDelay(2 * VAG_DISCONNECT_TIMEOUT);

		return status;
	}

	// Skipping VAG_BTI_INFO_RES blocks

	while (true)
	{
		VAG_BLOCK* pvb;
		MC_STATUS status = McReceiveBlock(&pvb);

		if (status != MC_S_OK)
		{
			McDelay(2 * VAG_DISCONNECT_TIMEOUT);

			return status;
		}

		if (pvb->blockheader.bBlockIndex != ++ g_bBlockIndex)
		{
			McMessage(MC_ML_ERROR, "Bad block index: '%02X' vs '%02X'\n", (unsigned) 
				pvb->blockheader.bBlockIndex, (unsigned) g_bBlockIndex);

			McFreeBlock(pvb);
			McDelay(2 * VAG_DISCONNECT_TIMEOUT);

			return MC_S_ERROR_CONNECT_BLOCK_INDEX;
		}

		if (pvb->blockheader.bBlockTypeId == VAG_BTI_ERROR)
		{
			McFreeBlock(pvb);
			McDelay(2 * VAG_DISCONNECT_TIMEOUT);

			return MC_S_ERROR_ECU;
		}

		if (pvb->blockheader.bBlockTypeId == VAG_BTI_ACK)
		{
			// VAG_BTI_ACK - no more descriptions

			McFreeBlock(pvb);
		
			break;
		}

		if (pvb->blockheader.bBlockTypeId != VAG_BTI_INFO_RES)
		{
			McMessage(MC_ML_ERROR, "Bad block type id: '%02X' vs '%02X'\n", (unsigned) 
				pvb->blockheader.bBlockTypeId, (unsigned) VAG_BTI_INFO_RES);

			McFreeBlock(pvb);
			McDelay(2 * VAG_DISCONNECT_TIMEOUT);

			return MC_S_ERROR_CONNECT;
		}

		McFreeBlock(pvb);

		VAG_BLOCK_ACK* pvba = (VAG_BLOCK_ACK*) McAllocBlock(sizeof(VAG_BLOCK_ACK));
		pvba->blockheader.bBlockIndex = ++ g_bBlockIndex;
		pvba->blockheader.bBlockTypeId = VAG_BTI_ACK;

		status = McSendBlock(pvba);

		if (status != MC_S_OK)
		{
			McFreeBlock(pvba);
			McDelay(2 * VAG_DISCONNECT_TIMEOUT);

			return status;
		}

		McFreeBlock(pvba);
	}

	++ g_cs.dwReestablishmentCount;

	return MC_S_OK;
}

// Allocates memory for a block and sets its size

VAG_BLOCK* McAllocBlock(BYTE bSize)
{
	assert(g_pvb == NULL || ! g_pvb->blockheader.bBlockSize);
	assert(bSize >= sizeof(VAG_BLOCK));

	if (bSize > g_bBlockSize)
	{
		// not realloc()
		free(g_pvb);

		g_pvb = (VAG_BLOCK*) malloc(bSize);
		g_bBlockSize = bSize;
	}

	g_pvb->blockheader.bBlockSize = bSize;

	return g_pvb;
}

// Frees memory of a block

void McFreeBlock(VAG_BLOCK* pvb)
{
	assert(pvb == g_pvb);
	assert(g_pvb->blockheader.bBlockSize);

	g_pvb->blockheader.bBlockSize = 0;	// freed sign
}

MC_STATUS McSetPortTimeout(DWORD dwReadAddTimeout, DWORD dwReadMultTimeout)
{
	COMMTIMEOUTS commtimouts = {MAXDWORD, dwReadMultTimeout, dwReadAddTimeout, 0, 0};

	BOOL b = SetCommTimeouts(g_hPort, &commtimouts);

	if (! b)
		return MC_S_ERROR_CONNECT_PORT;

	return MC_S_OK;
}

// Writes bytes to the port, doesn't require any echos

MC_STATUS McWritePort0(DWORD dwCountToWrite, const VOID* pvBuffer)
{
	assert(g_hPort != INVALID_HANDLE_VALUE);

	BOOL b = PurgeComm(g_hPort, PURGE_TXCLEAR);

	if (! b)
		return MC_S_ERROR_CONNECT_PORT;

	if (g_ml >= MC_ML_LOG_1)
		McMessage(MC_ML_LOG_1, "<<< [%u] %s %s", (unsigned) dwCountToWrite, 
			McGetHexStr(pvBuffer, dwCountToWrite), McGetTimeStr());

	DWORD dwCountWritten;
	b = WriteFile(g_hPort, pvBuffer, dwCountToWrite, &dwCountWritten, NULL);

	if (! b)
	{
		McMessage(MC_ML_LOG_1, " - Failed\n");

		return MC_S_ERROR_CONNECT_PORT;
	}

	if (g_ml >= MC_ML_LOG_1)
		McMessage(MC_ML_LOG_1, " - Ok [%u] %s\n", (unsigned) dwCountWritten, McGetTimeStr());

	if (dwCountWritten < dwCountToWrite)
		return MC_S_ERROR_CONNECT_PORT;

	return MC_S_OK;
}

// Writes bytes to the port, requires adapter echo
//
// We don't care much about the adapter echo, it just should be.

MC_STATUS McWritePort1(DWORD dwCountToWrite, const VOID* pvBuffer)
{
	assert(g_hPort != INVALID_HANDLE_VALUE);

	BOOL b = PurgeComm(g_hPort, PURGE_TXCLEAR | PURGE_RXCLEAR);

	if (! b)
		return MC_S_ERROR_CONNECT_PORT;

	for (DWORD dwIndex = 0; dwIndex < dwCountToWrite; ++ dwIndex)
	{
		BYTE bData = ((BYTE*) pvBuffer)[dwIndex];

		if (g_ml >= MC_ML_LOG_1)
		{
			if (dwIndex && g_ml >= MC_ML_LOG_2)
				McMessage(MC_ML_LOG_2, "\n");

			McMessage(MC_ML_LOG_1, "<<< [1] '%02X' %s", (unsigned) bData, McGetTimeStr());
		}

		DWORD dwCountWritten;
		b = WriteFile(g_hPort, &bData, 1, &dwCountWritten, NULL);

		if (! b)
		{
			McMessage(MC_ML_LOG_1, " - Failed\n");

			return MC_S_ERROR_CONNECT_PORT;
		}

		if (g_ml >= MC_ML_LOG_1)
			McMessage(MC_ML_LOG_1, " - Ok [%u] %s\n", (unsigned) dwCountWritten, McGetTimeStr());

		if (dwCountWritten < 1)
			return MC_S_ERROR_CONNECT_PORT;

		if (g_ml >= MC_ML_LOG_2)
			McMessage(MC_ML_LOG_2, ">>> [1] %s", McGetTimeStr());

		BYTE bDataEcho;
		DWORD dwCountRead;
		b = ReadFile(g_hPort, &bDataEcho, 1, &dwCountRead, NULL);

		if (! b)
		{
			McMessage(MC_ML_LOG_2, " - Failed\n");

			return MC_S_ERROR_CONNECT_PORT;
		}

		if (g_ml >= MC_ML_LOG_2)
			McMessage(MC_ML_LOG_2, " - Ok [%u] %s %s\n", (unsigned) dwCountRead,
				McGetHexStr(&bDataEcho, dwCountRead), McGetTimeStr());

		if (dwCountRead < 1)
		{
			McMessage(MC_ML_ERROR, "No adapter echo\n");

			return MC_S_ERROR_CONNECT_ADAPTER;
		}

		if (bDataEcho != bData)
		{
			McMessage(MC_ML_ERROR, "Bad adapter echo: '%02X' vs '%02X'\n", 
				(unsigned) bDataEcho, (unsigned) bData);
		}
	}

	return MC_S_OK;
}

// Writes bytes to the port, requires adapter echo and ECU echo
//
// We don't care much about the adapter echo, it just should be.

MC_STATUS McWritePort2(DWORD dwCountToWrite, const VOID* pvBuffer)
{
	assert(g_hPort != INVALID_HANDLE_VALUE);

	BOOL b = PurgeComm(g_hPort, PURGE_TXCLEAR | PURGE_RXCLEAR);

	if (! b)
		return MC_S_ERROR_CONNECT_PORT;

	for (DWORD dwIndex = 0; dwIndex < dwCountToWrite; ++ dwIndex)
	{
		BYTE bData = ((BYTE*) pvBuffer)[dwIndex];

		if (g_ml >= MC_ML_LOG_1)
		{
			if (dwIndex && g_ml >= MC_ML_LOG_2)
				McMessage(MC_ML_LOG_2, "\n");

			McMessage(MC_ML_LOG_1, "<<< [1] '%02X' %s", (unsigned) bData, McGetTimeStr());
		}

		DWORD dwCountWritten;
		b = WriteFile(g_hPort, &bData, 1, &dwCountWritten, NULL);

		if (! b)
		{
			McMessage(MC_ML_LOG_1, " - Failed\n");

			return MC_S_ERROR_CONNECT_PORT;
		}

		if (g_ml >= MC_ML_LOG_1)
			McMessage(MC_ML_LOG_1, " - Ok [%u] %s\n", (unsigned) dwCountWritten, McGetTimeStr());

		if (dwCountWritten < 1)
			return MC_S_ERROR_CONNECT_PORT;

		if (g_ml >= MC_ML_LOG_2)
			McMessage(MC_ML_LOG_2, ">>> [2] %s", McGetTimeStr());

		BYTE abDataEcho[2];
		DWORD dwCountRead;
		b = ReadFile(g_hPort, &abDataEcho, 2, &dwCountRead, NULL);

		if (! b)
		{
			McMessage(MC_ML_LOG_2, " - Failed\n");

			return MC_S_ERROR_CONNECT_PORT;
		}

		if (g_ml >= MC_ML_LOG_2)
			McMessage(MC_ML_LOG_2, " - Ok [%u] %s %s\n", (unsigned) dwCountRead,
				McGetHexStr(abDataEcho, dwCountRead), McGetTimeStr());

		if (dwCountRead < 1)
		{
			McMessage(MC_ML_ERROR, "No adapter echo\n");

			return MC_S_ERROR_CONNECT_ADAPTER;
		}

		if (abDataEcho[0] != bData)
		{
			McMessage(MC_ML_ERROR, "Bad adapter echo: '%02X' vs '%02X'\n", 
				(unsigned) abDataEcho[0], (unsigned) bData);
		}

		if (dwCountRead < 2)
		{
			McMessage(MC_ML_ERROR, "No ECU echo\n");

			return MC_S_ERROR_CONNECT_ECU;
		}

#ifdef MC_EMULATE_COMM_ERRORS

		static bool bTest = true;

		if (bData == 0x18 || bData == 0x19)
		{
			if (bTest)
			{
				McMessage(MC_ML_ERROR, "Simulating write error\n");
				abDataEcho[1] = bData;

				bTest = false;
			}
			else
				bTest = true;
		}
#endif
		if (abDataEcho[1] != (BYTE) ~bData)
		{
			McMessage(MC_ML_ERROR, "Bad ECU echo: '%02X' vs '%02X'\n", 
				(unsigned) abDataEcho[1], (unsigned) (BYTE) ~bData);

			return MC_S_ERROR_CONNECT_ECU_ECHO;
		}
	}

	return MC_S_OK;
}

// Reads up to dwCountToRead bytes from the port, does't echo

MC_STATUS McReadPort0(DWORD dwCountToRead, VOID* pvBuffer, DWORD& rdwCountRead)
{
	assert(g_hPort != INVALID_HANDLE_VALUE);

	if (g_ml >= MC_ML_LOG_1)
		McMessage(MC_ML_LOG_1, ">>> [%u] %s", (unsigned) dwCountToRead, McGetTimeStr());

	BOOL b = ReadFile(g_hPort, pvBuffer, dwCountToRead, &rdwCountRead, NULL);

	if (! b)
	{
		McMessage(MC_ML_LOG_1, " - Failed\n");

		return MC_S_ERROR_CONNECT_PORT;
	}

	if (g_ml >= MC_ML_LOG_1)
		McMessage(MC_ML_LOG_1, " - Ok [%u] %s %s\n", (unsigned) rdwCountRead,
			McGetHexStr(pvBuffer, rdwCountRead), McGetTimeStr());

	return MC_S_OK;
}

// Reads bytes from the port, does't echo

MC_STATUS McReadPort0(DWORD dwCountToRead, VOID* pvBuffer)
{
	assert(g_hPort != INVALID_HANDLE_VALUE);

	if (g_ml >= MC_ML_LOG_1)
		McMessage(MC_ML_LOG_1, ">>> [%u] %s", (unsigned) dwCountToRead, McGetTimeStr());

	DWORD dwCountRead;
	BOOL b = ReadFile(g_hPort, pvBuffer, dwCountToRead, &dwCountRead, NULL);

	if (! b)
	{
		McMessage(MC_ML_LOG_1, " - Failed\n");

		return MC_S_ERROR_CONNECT_PORT;
	}

	if (g_ml >= MC_ML_LOG_1)
		McMessage(MC_ML_LOG_1, " - Ok [%u] %s %s\n", (unsigned) dwCountRead,
			McGetHexStr(pvBuffer, dwCountRead), McGetTimeStr());

	if (dwCountRead < dwCountToRead)
	{
		McMessage(MC_ML_ERROR, "No ECU bytes: %u vs %u\n", (unsigned) dwCountRead,
			(unsigned) dwCountToRead);

		return MC_S_ERROR_CONNECT_ECU;
	}

	return MC_S_OK;
}

// Reads bytes from the port, echos, requires adapter echo
//
// We don't care much about the adapter echo, it just should be.

MC_STATUS McReadPort2(DWORD dwCountToRead, VOID* pvBuffer)
{
	assert(g_hPort != INVALID_HANDLE_VALUE);

	BOOL b = PurgeComm(g_hPort, PURGE_TXCLEAR);

	if (! b)
		return MC_S_ERROR_CONNECT_PORT;

	for (DWORD dwIndex = 0; dwIndex < dwCountToRead; ++ dwIndex)
	{
		if (g_ml >= MC_ML_LOG_1)
		{
			if (dwIndex && g_ml >= MC_ML_LOG_2)
				McMessage(MC_ML_LOG_2, "\n");

			McMessage(MC_ML_LOG_1, ">>> [1] %s", McGetTimeStr());
		}

		BYTE bData;
		DWORD dwCountRead;
		b = ReadFile(g_hPort, &bData, 1, &dwCountRead, NULL);

		if (! b)
		{
			McMessage(MC_ML_LOG_1, " - Failed\n");

			return MC_S_ERROR_CONNECT_PORT;
		}

		if (g_ml >= MC_ML_LOG_1)
			McMessage(MC_ML_LOG_1, " - Ok [%u] %s %s\n", (unsigned) dwCountRead,	
				McGetHexStr(&bData, dwCountRead), McGetTimeStr());

		if (dwCountRead < 1)
		{
			McMessage(MC_ML_ERROR, "No ECU bytes: %u vs 1\n", (unsigned) dwCountRead);

			return MC_S_ERROR_CONNECT_ECU;
		}

		((BYTE*) pvBuffer)[dwIndex] = bData;

		BYTE bDataEcho = ~bData;

#ifdef MC_EMULATE_COMM_ERRORS

		static bool bTest = true;

		if (bData == 0x28 || bData == 0x29)
		{
			if (bTest)
			{
				McMessage(MC_ML_ERROR, "Simulating read error\n");
				bDataEcho = bData;

				bTest = false;
			}
			else
				bTest = true;
		}
#endif
		if (g_ml >= MC_ML_LOG_2)
			McMessage(MC_ML_LOG_2, "<<< [1] '%02X' %s", (unsigned) bDataEcho, McGetTimeStr());

		DWORD dwCountWritten;
		b = WriteFile(g_hPort, &bDataEcho, 1, &dwCountWritten, NULL);

		if (! b)
		{
			McMessage(MC_ML_LOG_2, " - Failed\n");

			return MC_S_ERROR_CONNECT_PORT;
		}

		if (g_ml >= MC_ML_LOG_2)
			McMessage(MC_ML_LOG_2, " - Ok [%u] %s\n", (unsigned) dwCountWritten, McGetTimeStr());

		if (dwCountWritten < 1)
			return MC_S_ERROR_CONNECT_PORT;

		if (g_ml >= MC_ML_LOG_2)
			McMessage(MC_ML_LOG_2, ">>> [1] %s", McGetTimeStr());

		BYTE bDataEchoEcho;
		b = ReadFile(g_hPort, &bDataEchoEcho, 1, &dwCountRead, NULL);

		if (! b)
		{
			McMessage(MC_ML_LOG_2, " - Failed\n");

			return MC_S_ERROR_CONNECT_PORT;
		}

		if (g_ml >= MC_ML_LOG_2)
			McMessage(MC_ML_LOG_2, " - Ok [%u] %s %s\n", (unsigned) dwCountRead,	
				McGetHexStr(&bDataEchoEcho, dwCountRead), McGetTimeStr());

		if (dwCountRead < 1)
		{
			McMessage(MC_ML_ERROR, "No adapter echo\n");

			return MC_S_ERROR_CONNECT_ADAPTER;
		}

		if (bDataEchoEcho != bDataEcho)
		{
			McMessage(MC_ML_ERROR, "Bad adapter echo: '%02X' vs '%02X'\n", 
				(unsigned) bDataEchoEcho, (unsigned) bDataEcho);
		}
	}

	return MC_S_OK;
}

MC_STATUS McBreakPort()
{
	// We don't just wait for the required time here, but try to verify the
	// line state - noise, noise...	

	MC_STATUS status = McSetPortTimeout(2 * VAG_BREAK_TIMEOUT, 0);

	if (status != MC_S_OK)
		return status;

	BOOL b = PurgeComm(g_hPort, PURGE_RXCLEAR);

	if (! b)
		return MC_S_ERROR_CONNECT_PORT;

	McMessage(MC_ML_LOG_2, "Breaking %s", McGetTimeStr());

	BYTE ab[16];
	DWORD dwCountRead;
	b = ReadFile(g_hPort, ab, sizeof ab, &dwCountRead, NULL);

	if (! b)
	{
		McMessage(MC_ML_LOG_2, " - Failed\n");

		return MC_S_ERROR_CONNECT_PORT;
	}

	if (dwCountRead)
	{
		// We couldn't hold the line off... (noise bytes, out of sych ECU bytes, etc)

		McMessage(MC_ML_LOG_2, " - Failed [%u] %s %s\n", (unsigned) dwCountRead, 
			McGetHexStr(ab, dwCountRead), McGetTimeStr());

		return MC_S_ERROR_CONNECT;
	}

	// Sounds not bad

	McMessage(MC_ML_LOG_2, " - Ok %s\n", McGetTimeStr());

	return MC_S_OK;
}

void McMessageConnectionStatistics(MC_MESSAGE_LEVEL ml)
{
	DWORD dwDuration = (g_cs.dwEndTickCount - g_cs.dwStartTickCount) / 1000;

	McMessage(ml, "\nConnection duration - %u sec\n"
		"Blocks sent - %u, breaks - %u\n"
		"Blocks received - %u, resets - %u\n"
		"Reestablishments - %u\n",
		(unsigned) dwDuration,
		(unsigned) g_cs.dwBlockSentCount, (unsigned) g_cs.dwBreakCount,
		(unsigned) g_cs.dwBlockReceivedCount, (unsigned) g_cs.dwResetCount,
		(unsigned) g_cs.dwReestablishmentCount);
}

// Consider checking g_ml before calling McMessage() in time critical
// operations to avoid unnecessary aruments preparing and function calls

void McMessage(MC_MESSAGE_LEVEL ml, LPCSTR pszFormat, ...)
{	
	if (g_mh == NULL || ml > g_ml)
		return;

	va_list list;
	va_start(list, pszFormat);

	if (g_bMessagesSuspended)
	{
		assert(MC_MAX_MESSAGE_SIZE < 0x1000);

		if (g_dwMessageCapacity - g_dwMessageLength < MC_MAX_MESSAGE_SIZE)
		{
			g_pszMessage = (LPSTR) realloc(g_pszMessage, g_dwMessageCapacity + 0x1000);
			g_dwMessageCapacity += 0x1000;
		}

		LPSTR pszMessage = g_pszMessage + g_dwMessageLength;

		DWORD dwMaxCount = g_dwMessageCapacity - g_dwMessageLength;
		int iCount = _vsnprintf(pszMessage, dwMaxCount, pszFormat, list);

		if (iCount == -1 || iCount == dwMaxCount)
		{
			pszMessage[dwMaxCount - 1] = '\0';
			pszMessage[dwMaxCount - 2] = '.';
			pszMessage[dwMaxCount - 3] = '.';
			pszMessage[dwMaxCount - 4] = '.';

			iCount = dwMaxCount - 1;
		}

		g_dwMessageLength += iCount;
	}
	else
	{
		char szMessage[MC_MAX_MESSAGE_SIZE];
		int iCount = _vsnprintf(szMessage, MC_MAX_MESSAGE_SIZE, pszFormat, list);

		if (iCount == -1 || iCount == MC_MAX_MESSAGE_SIZE)
		{
			szMessage[MC_MAX_MESSAGE_SIZE - 1] = '\0';
			szMessage[MC_MAX_MESSAGE_SIZE - 2] = '.';
			szMessage[MC_MAX_MESSAGE_SIZE - 3] = '.';
			szMessage[MC_MAX_MESSAGE_SIZE - 4] = '.';
		}

		__try
		{
			g_mh(g_pvContext, szMessage);
		}
		__except (EXCEPTION_EXECUTE_HANDLER)
		{
		}
	}

	va_end(list);
}

// Use McSuspendMessages() to suspend logging and accumulate log messages. McResumeMessages() 
// yields the accumulated log message and resumes logging.

void McSuspendMessages()
{
	assert(! g_bMessagesSuspended);

	g_dwMessageLength = 0;

	g_bMessagesSuspended = true;	
}

void McResumeMessages()
{
	assert(g_bMessagesSuspended);

	if (g_dwMessageLength)
	{
		__try
		{
			g_mh(g_pvContext, g_pszMessage);
		}
		__except (EXCEPTION_EXECUTE_HANDLER)
		{
		}
	}

	g_bMessagesSuspended = false;
}

CHAR McGetHexChar(BYTE bDigit)
{
	return bDigit < 0xA ? '0' + bDigit : 'A' + (bDigit - 0xA);
}

LPCSTR McGetHexStr(LPCVOID lpBuffer, DWORD dwCount)
{
	static CHAR sz[MC_MAX_MESSAGE_SIZE];

	CHAR* pc = sz, *pcEnd = sz + MC_MAX_MESSAGE_SIZE - 2;
	BYTE* pb = (BYTE*) lpBuffer, *pbEnd = (BYTE*) lpBuffer + dwCount;

	*pc ++ = '\'';

	while (pb != pbEnd)
	{
		if (pc + 3 > pcEnd)
		{
			pc[-1] = '.';
			pc[-2] = '.';
			pc[-3] = '.';

			break;
		}

		BYTE b = *pb ++;

		if (pc != sz + 1)
		{
			*pc ++ = ' ';
		}

		*pc ++ = McGetHexChar(b >> 4);
		*pc ++ = McGetHexChar(b & 0xF);
	}

	*pc ++ = '\'';
	*pc ++ = '\0';

	return sz;
}

LPCSTR McGetTimeStr()
{
	LARGE_INTEGER liCount;
	BOOL b = QueryPerformanceCounter(&liCount);

	if (! b)
		return "";

	static CHAR sz[128];

	_snprintf(sz, sizeof sz, "@ %.1f ms", g_dTimerResolution * (liCount.QuadPart - g_liStartCount.QuadPart));

	return sz;
}

void McDelay(DWORD dwMillisec)
{
	Sleep(dwMillisec);
}

MC_STATUS McBeginTimeCriticalOperation()
{
	// We don't want our code to be interrupted not to produce FALSE break events 
	// during trasmission and not to MISS break events during reception (!)

	// We don't want client arbitrary code to be executed in this thread 
	McSuspendMessages();

	// We don't want this thread to be suspended in favour of another thread
	MC_STATUS status = McSetTimeCriticalThreadPriority();

	if (status != MC_S_OK)
	{
		McResumeMessages();

		return status;
	}

	return MC_S_OK;
}

void McEndTimeCriticalOperation()
{
	McRestoreThreadPriority();

	McResumeMessages();
}

MC_STATUS McSetTimeCriticalThreadPriority()
{
	assert(g_iThreadPriorityOld == THREAD_PRIORITY_ERROR_RETURN);

	HANDLE hCurrentThread = GetCurrentThread();

	g_iThreadPriorityOld = GetThreadPriority(hCurrentThread);

	if (g_iThreadPriorityOld == THREAD_PRIORITY_ERROR_RETURN)
		return MC_S_ERROR;

	BOOL b = SetThreadPriority(hCurrentThread, THREAD_PRIORITY_TIME_CRITICAL);

	if (! b)
	{
		g_iThreadPriorityOld = THREAD_PRIORITY_ERROR_RETURN;

		return MC_S_ERROR;
	}

	return MC_S_OK;
}

void McRestoreThreadPriority()
{
	assert(g_iThreadPriorityOld != THREAD_PRIORITY_ERROR_RETURN);

	SetThreadPriority(GetCurrentThread(), g_iThreadPriorityOld);
	g_iThreadPriorityOld = THREAD_PRIORITY_ERROR_RETURN;
}